<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

requireLogin();
$activePage = 'quotation';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Check permissions
$isSuperAdmin = false;
if (isset($_SESSION['user_id'])) {
    $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = (SELECT role_id FROM users WHERE id = ?)");
    $stmt->execute([$_SESSION['user_id']]);
    $isSuperAdmin = $stmt->fetchColumn();
}

$userName = $_SESSION['username'] ?? 'User';

// Get store settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
$stmt->execute();
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get quotations (handle case where table doesn't exist)
$quotations = [];
try {
    $stmt = $pdo->prepare("
        SELECT 
            q.*,
            c.name as customer_name,
            u.username as created_by_name
        FROM quotations q
        LEFT JOIN customers c ON q.customer_id = c.id
        JOIN users u ON q.created_by = u.id
        ORDER BY q.created_at DESC
        LIMIT 50
    ");
    $stmt->execute();
    $quotations = $stmt->fetchAll();
} catch (PDOException $e) {
    // Table doesn't exist yet - will be created later
    $quotations = [];
}

// Add search bar above quotations table
if (isset($_GET['search']) && $_GET['search'] !== '') {
    $search = strtolower(trim($_GET['search']));
    $quotations = array_filter($quotations, function($q) use ($search) {
        return strpos(strtolower($q['customer_name']), $search) !== false
            || strpos(strtolower($q['quote_number']), $search) !== false
            || strpos(strtolower($q['status']), $search) !== false;
    });
}

// Get customers for dropdown
$stmt = $pdo->prepare("SELECT id, name FROM customers ORDER BY name");
$stmt->execute();
$customers = $stmt->fetchAll();

// Get products for quotation items
$stmt = $pdo->prepare("SELECT id, name, price FROM products ORDER BY name");
$stmt->execute();
$products = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Quotations - <?php echo htmlspecialchars($settings['store_name'] ?? 'POS System'); ?></title>
    
    <!-- Favicon -->
    <?php echo getFaviconLink($pdo); ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Custom CSS -->
    <link href="../assets/css/style.css" rel="stylesheet">
    
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    
    <style>
        :root {
            --primary-color: #2563eb;
            --primary-dark: #1d4ed8;
            --secondary-color: #64748b;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #06b6d4;
            --light-bg: #f8fafc;
            --border-color: #e2e8f0;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
        }

        * {
            font-family: 'Inter', sans-serif;
        }

        body {
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%);
            color: var(--text-primary);
            min-height: 100vh;
        }

        .main-content {
            margin-left: 280px;
            padding: 2rem;
            transition: all 0.3s ease;
        }
        
        @media (max-width: 991px) {
            .main-content {
                margin-left: 0;
                padding: 1rem;
                padding-top: 80px;
            }
        }

        /* Header Section */
        .page-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 20px;
            padding: 1rem;
            margin-bottom: 2rem;
            color: white;
            box-shadow: var(--shadow-lg);
            position: relative;
            overflow: hidden;
        }

        .page-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/><circle cx="10" cy="60" r="0.5" fill="white" opacity="0.1"/><circle cx="90" cy="40" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
            opacity: 0.3;
        }

        .page-header h1 {
            font-size: 1.75rem;
            font-weight: 700;
            margin: 0;
            position: relative;
            z-index: 1;
        }

        .page-header p {
            font-size: 0.9rem;
            opacity: 0.9;
            margin: 0.25rem 0 0 0;
            position: relative;
            z-index: 1;
        }

        .header-actions {
            position: relative;
            z-index: 1;
        }

        /* Cards */
        .card {
            border: none;
            border-radius: 16px;
            box-shadow: var(--shadow-md);
            background: white;
            transition: all 0.3s ease;
            overflow: hidden;
        }

        .card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }
        
        .card-header {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border-bottom: 1px solid var(--border-color);
            padding: 0.75rem;
            border-radius: 16px 16px 0 0 !important;
        }

        .card-header h5 {
            font-weight: 600;
            color: var(--text-primary);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1rem;
        }

        .card-body {
            padding: 0.75rem;
        }

        /* Buttons */
        .btn {
            border-radius: 12px;
            font-weight: 500;
            padding: 0.75rem 1.5rem;
            transition: all 0.3s ease;
            border: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 99, 235, 0.3);
        }

        .btn-success {
            background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);
            color: white;
        }

        .btn-warning {
            background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%);
            color: white;
        }

        .btn-danger {
            background: linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%);
            color: white;
        }

        .btn-info {
            background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%);
            color: white;
        }

        .btn-outline-primary {
            border: 2px solid var(--primary-color);
            color: var(--primary-color);
            background: transparent;
        }

        .btn-outline-primary:hover {
            background: var(--primary-color);
            color: white;
            transform: translateY(-2px);
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            border-radius: 8px;
        }

        /* Search Bar */
        .search-container {
            position: relative;
            max-width: 400px;
        }

        .search-container .form-control {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 0.75rem 1rem 0.75rem 3rem;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: white;
        }

        .search-container .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .search-container .search-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
            z-index: 2;
        }

        /* Table */
        .table {
            margin: 0;
            border-radius: 12px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border: none;
            padding: 0.5rem;
            font-weight: 600;
            color: var(--text-primary);
            text-transform: uppercase;
            font-size: 0.75rem;
            letter-spacing: 0.5px;
        }

        .table td {
            padding: 0.5rem;
            border: none;
            border-bottom: 1px solid var(--border-color);
            vertical-align: middle;
        }

        .table tbody tr {
            transition: all 0.3s ease;
        }

        .table tbody tr:hover {
            background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%);
            transform: scale(1.01);
        }

        .table tbody tr:last-child td {
            border-bottom: none;
        }

        /* Badges */
        .badge {
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-weight: 500;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .bg-success {
            background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%) !important;
        }

        .bg-danger {
            background: linear-gradient(135deg, var(--danger-color) 0%, #dc2626 100%) !important;
        }

        .bg-warning {
            background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%) !important;
            color: white !important;
        }

        .bg-info {
            background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%) !important;
        }

        /* Action Buttons */
        .btn-group {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }

        .btn-group .btn {
            border-radius: 8px;
            padding: 0.5rem;
            min-width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 3rem 2rem;
            color: var(--text-secondary);
        }

        .empty-state i {
            font-size: 4rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        .empty-state h5 {
            font-size: 1.25rem;
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }

        .empty-state p {
            font-size: 1rem;
            margin: 0;
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 0.375rem;
            box-shadow: var(--shadow-md);
            transition: all 0.3s ease;
            border-left: 4px solid var(--primary-color);
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .stat-card .stat-icon {
            width: 25px;
            height: 25px;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.25rem;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
            font-size: 0.75rem;
        }

        .stat-card .stat-value {
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.0625rem;
        }

        .stat-card .stat-label {
            color: var(--text-secondary);
            font-size: 0.625rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Modal Enhancements */
        .modal-content {
            border: none;
            border-radius: 20px;
            box-shadow: var(--shadow-lg);
        }

        .modal-header {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            border-bottom: 1px solid var(--border-color);
            border-radius: 20px 20px 0 0;
            padding: 1.5rem;
        }

        .modal-title {
            font-weight: 600;
            color: var(--text-primary);
        }

        .modal-body {
            padding: 1.5rem;
        }

        .modal-footer {
            border-top: 1px solid var(--border-color);
            padding: 1.5rem;
            border-radius: 0 0 20px 20px;
        }

        /* Form Controls */
        .form-control, .form-select {
            border-radius: 12px;
            border: 2px solid var(--border-color);
            padding: 0.75rem 1rem;
            transition: all 0.3s ease;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .form-label {
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .page-header h1 {
                font-size: 2rem;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .btn-group {
                flex-direction: column;
            }
            
            .btn-group .btn {
                width: 100%;
            }
        }

        /* Loading Animation */
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        /* Alert Enhancements */
        .alert {
            border: none;
            border-radius: 12px;
            padding: 1rem 1.5rem;
            margin-bottom: 1rem;
            box-shadow: var(--shadow-sm);
        }

        .alert-success {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            color: #065f46;
        }

        .alert-danger {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            color: #991b1b;
        }

        .alert-warning {
            background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
            color: #92400e;
        }

        .alert-info {
            background: linear-gradient(135deg, #cffafe 0%, #a5f3fc 100%);
            color: #155e75;
        }
    </style>
</head>
<body>
    <!-- Include Sidebar -->
    <?php include '../includes/sidebar.php'; ?>
    <?php include '../includes/topbar.php'; ?>
    
    <!-- Alert Container -->
    <div id="alertContainer" class="position-fixed top-0 end-0 p-3" style="z-index: 9999;"></div>
    
    <div class="main-content">
        <!-- Page Header -->
        <div class="page-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-file-invoice me-3"></i>Quotations</h1>
                    <p>Manage and track all your business quotations</p>
                </div>
                <div class="header-actions">
            <?php if (hasPermission('add_quotation') || hasPermission('view_quotation')): ?>
                    <button class="btn btn-light" data-bs-toggle="modal" data-bs-target="#addQuotationModal">
                        <i class="fas fa-plus"></i>
                        Create Quotation
            </button>
            <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Stats Cards -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-file-invoice"></i>
                </div>
                <div class="stat-value"><?php echo count($quotations); ?></div>
                <div class="stat-label">Total Quotations</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--success-color) 0%, #059669 100%);">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="stat-value"><?php echo count(array_filter($quotations, fn($q) => $q['status'] === 'active')); ?></div>
                <div class="stat-label">Active Quotations</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--warning-color) 0%, #d97706 100%);">
                    <i class="fas fa-clock"></i>
                </div>
                <div class="stat-value"><?php echo count(array_filter($quotations, fn($q) => $q['status'] === 'expired')); ?></div>
                <div class="stat-label">Expired Quotations</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon" style="background: linear-gradient(135deg, var(--info-color) 0%, #0891b2 100%);">
                    <i class="fas fa-truck"></i>
                </div>
                <div class="stat-value"><?php echo count(array_filter($quotations, fn($q) => $q['status'] === 'delivery_scheduled')); ?></div>
                <div class="stat-label">Scheduled Deliveries</div>
            </div>
        </div>
        
        <!-- Quotations Table -->
        <div class="card">
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h5>
                        <i class="fas fa-list"></i>
                        All Quotations
                </h5>
                    <div class="search-container">
                        <i class="fas fa-search search-icon"></i>
                        <input type="text" class="form-control" id="quotationSearchInput" placeholder="Search quotations...">
            </div>
                </div>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Quote #</th>
                                <th>Customer</th>
                                <th>Total Amount</th>
                                <th>Status</th>
                                <th>Created Date</th>
                                <th>Valid Until</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($quotations)): ?>
                                <tr>
                                    <td colspan="7">
                                        <div class="empty-state">
                                            <i class="fas fa-file-invoice"></i>
                                            <h5>No quotations found</h5>
                                            <p>Create your first quotation to get started!</p>
                                            <button class="btn btn-primary mt-3" data-bs-toggle="modal" data-bs-target="#addQuotationModal">
                                                <i class="fas fa-plus me-2"></i>Create Quotation
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($quotations as $quotation): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($quotation['quote_number']); ?></strong>
                                        </td>
                                        <td>
                                            <div>
                                                <strong><?php echo htmlspecialchars($quotation['customer_name'] ?? 'Walk-in Customer'); ?></strong>
                                                <br>
                                                <small class="text-muted">Created by <?php echo htmlspecialchars($quotation['created_by_name']); ?></small>
                                            </div>
                                        </td>
                                        <td>
                                            <strong class="text-primary">Ksh <?php echo number_format($quotation['total_amount'], 2); ?></strong>
                                        </td>
                                        <td>
                                            <span class="badge <?php 
                                                echo match($quotation['status']) {
                                                    'active' => 'bg-success',
                                                    'expired' => 'bg-danger',
                                                    'delivery_scheduled' => 'bg-info',
                                                    default => 'bg-warning'
                                                };
                                            ?>">
                                                <i class="fas fa-<?php 
                                                    echo match($quotation['status']) {
                                                        'active' => 'check-circle',
                                                        'expired' => 'times-circle',
                                                        'delivery_scheduled' => 'truck',
                                                        default => 'clock'
                                                    };
                                                ?> me-1"></i>
                                                <?php echo ucfirst(str_replace('_', ' ', $quotation['status'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div>
                                                <strong><?php echo date('M j, Y', strtotime($quotation['created_at'])); ?></strong>
                                                <br>
                                                <small class="text-muted"><?php echo date('g:i A', strtotime($quotation['created_at'])); ?></small>
                                            </div>
                                        </td>
                                        <td>
                                            <div>
                                                <strong><?php echo date('M j, Y', strtotime($quotation['valid_until'])); ?></strong>
                                                <br>
                                                <small class="text-muted">
                                                    <?php 
                                                    $daysLeft = (strtotime($quotation['valid_until']) - time()) / (60 * 60 * 24);
                                                    if ($daysLeft > 0) {
                                                        echo round($daysLeft) . ' days left';
                                                    } else {
                                                        echo 'Expired';
                                                    }
                                                    ?>
                                                </small>
                                            </div>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <?php if (hasPermission('view_quotation')): ?>
                                                <button class="btn btn-outline-primary view-quotation" 
                                                        title="View Quotation" 
                                                        data-id="<?php echo $quotation['id']; ?>"
                                                        data-quote-number="<?php echo htmlspecialchars($quotation['quote_number']); ?>"
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#viewQuotationModal">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('edit_quotation')): ?>
                                                <button class="btn btn-outline-info edit-quotation" 
                                                        title="Edit Quotation"
                                                        data-id="<?php echo $quotation['id']; ?>"
                                                        data-quote-number="<?php echo htmlspecialchars($quotation['quote_number']); ?>">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('convert_quotation_to_invoice')): ?>
                                                <button class="btn btn-outline-success convert-to-sale" 
                                                        title="Convert to Sale"
                                                        data-id="<?php echo $quotation['id']; ?>"
                                                        data-quote-number="<?php echo htmlspecialchars($quotation['quote_number']); ?>"
                                                        data-total="<?php echo $quotation['total_amount']; ?>"
                                                        <?php echo $quotation['status'] !== 'active' ? 'disabled' : ''; ?>>
                                                    <i class="fas fa-arrow-right"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('schedule_delivery')): ?>
                                                <button class="btn btn-outline-warning schedule-delivery" 
                                                        title="Schedule Delivery"
                                                        data-id="<?php echo $quotation['id']; ?>"
                                                        data-quote-number="<?php echo htmlspecialchars($quotation['quote_number']); ?>"
                                                        data-customer-id="<?php echo $quotation['customer_id']; ?>"
                                                        data-customer-name="<?php echo htmlspecialchars($quotation['customer_name'] ?? 'Walk-in Customer'); ?>"
                                                        data-customer-phone="<?php echo htmlspecialchars($quotation['customer_phone'] ?? ''); ?>"
                                                        data-customer-address="<?php echo htmlspecialchars($quotation['customer_address'] ?? ''); ?>"
                                                        <?php echo $quotation['status'] !== 'active' ? 'disabled' : ''; ?>>
                                                    <i class="fas fa-truck"></i>
                                                </button>
                                                <?php endif; ?>
                                                <?php if (hasPermission('delete_quotation')): ?>
                                                <button class="btn btn-outline-danger delete-quotation" 
                                                        title="Delete Quotation"
                                                        data-id="<?php echo $quotation['id']; ?>"
                                                        data-quote-number="<?php echo htmlspecialchars($quotation['quote_number']); ?>">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Quotation Modal -->
    <div class="modal fade" id="addQuotationModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Create New Quotation</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="addQuotationForm">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="customer_id" class="form-label">Customer</label>
                                    <select class="form-select" id="customer_id" name="customer_id">
                                        <option value="">Walk-in Customer</option>
                                        <?php foreach ($customers as $customer): ?>
                                            <option value="<?php echo $customer['id']; ?>"><?php echo htmlspecialchars($customer['name']); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="valid_until" class="form-label">Valid Until</label>
                                    <input type="date" class="form-control" id="valid_until" name="valid_until" required>
                                </div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label for="notes" class="form-label">Notes</label>
                            <textarea class="form-control" id="notes" name="notes" rows="3" placeholder="Additional notes for the quotation..."></textarea>
                        </div>
                        
                        <!-- Product Selection -->
                        <div class="row mb-2">
                            <div class="col-md-4">
                                <h6 class="mb-0">Quotation Items</h6>
                            </div>
                            <div class="col-md-4">
                                <small class="text-muted" id="addScrollHint" style="display: none;">Scroll to see all items</small>
                            </div>
                            <div class="col-md-4">
                                <div class="position-relative">
                                    <input type="text" class="form-control form-control-sm" id="addProductSearch" placeholder="Search products to add..." autocomplete="off">
                                    <div id="addSearchResults" class="position-absolute w-100 bg-white border rounded shadow-sm" style="top: 100%; left: 0; z-index: 1000; max-height: 200px; overflow-y: auto; display: none;"></div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Fixed Header for Add Items -->
                        <div class="row mb-2">
                            <div class="col-md-5">
                                <small class="text-muted fw-bold">Product</small>
                            </div>
                            <div class="col-md-2">
                                <small class="text-muted fw-bold">Quantity</small>
                            </div>
                            <div class="col-md-2">
                                <small class="text-muted fw-bold">Unit Price</small>
                            </div>
                            <div class="col-md-2">
                                <small class="text-muted fw-bold">Total</small>
                            </div>
                            <div class="col-md-1">
                                <small class="text-muted fw-bold">Action</small>
                            </div>
                        </div>
                        
                        <div id="quotationItems" style="max-height: 280px; overflow-y: auto;">
                            <div class="row quotation-item mb-2">
                                <div class="col-md-5">
                                    <select class="form-select" name="product_id[]" required>
                                        <option value="">Select Product</option>
                                        <?php foreach ($products as $product): ?>
                                            <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price']; ?>">
                                                <?php echo htmlspecialchars($product['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="quantity[]" placeholder="Qty" min="1" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="unit_price[]" placeholder="Price" step="0.01" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="total_price[]" placeholder="Total" readonly>
                                </div>
                                <div class="col-md-1">
                                    <button type="button" class="btn btn-outline-danger btn-sm remove-item">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row mt-3">
                            <div class="col-md-8"></div>
                            <div class="col-md-4">
                                <div class="card">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between">
                                            <span>Total:</span>
                                            <span id="quotationTotal">Ksh 0.00</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Create Quotation</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View Quotation Modal -->
    <div class="modal fade" id="viewQuotationModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">View Quotation</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="quotationDetails">
                        <div class="text-center py-4">
                            <i class="fas fa-spinner fa-spin fa-2x text-muted"></i>
                            <p class="text-muted mt-2">Loading quotation details...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="printQuotation">
                        <i class="fas fa-print"></i> Print
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Quotation Modal -->
    <div class="modal fade" id="editQuotationModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Quotation</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form id="editQuotationForm">
                    <input type="hidden" id="edit_quotation_id" name="quotation_id">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_customer_id" class="form-label">Customer</label>
                                    <select class="form-select" id="edit_customer_id" name="customer_id">
                                        <option value="">Walk-in Customer</option>
                                        <?php foreach ($customers as $customer): ?>
                                            <option value="<?php echo $customer['id']; ?>"><?php echo htmlspecialchars($customer['name']); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_valid_until" class="form-label">Valid Until</label>
                                    <input type="date" class="form-control" id="edit_valid_until" name="valid_until" required>
                                </div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label for="edit_notes" class="form-label">Notes</label>
                            <textarea class="form-control" id="edit_notes" name="notes" rows="3" placeholder="Additional notes for the quotation..."></textarea>
                        </div>
                        
                        <!-- Product Selection -->
                        <div class="row mb-2">
                            <div class="col-md-4">
                                <h6 class="mb-0">Quotation Items</h6>
                            </div>
                            <div class="col-md-4">
                                <small class="text-muted" id="editScrollHint" style="display: none;">Scroll to see all items</small>
                            </div>
                            <div class="col-md-4">
                                <div class="position-relative">
                                    <input type="text" class="form-control form-control-sm" id="editProductSearch" placeholder="Search products to add..." autocomplete="off">
                                    <div id="editSearchResults" class="position-absolute w-100 bg-white border rounded shadow-sm" style="top: 100%; left: 0; z-index: 1000; max-height: 200px; overflow-y: auto; display: none;"></div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Fixed Header for Edit Items -->
                        <div class="row mb-2">
                            <div class="col-md-5">
                                <small class="text-muted fw-bold">Product</small>
                            </div>
                            <div class="col-md-2">
                                <small class="text-muted fw-bold">Quantity</small>
                            </div>
                            <div class="col-md-2">
                                <small class="text-muted fw-bold">Unit Price</small>
                            </div>
                            <div class="col-md-2">
                                <small class="text-muted fw-bold">Total</small>
                            </div>
                            <div class="col-md-1">
                                <small class="text-muted fw-bold">Action</small>
                            </div>
                        </div>
                        
                        <div id="editQuotationItems" style="max-height: 280px; overflow-y: auto;">
                            <!-- Items will be populated by JavaScript -->
                        </div>
                        

                        
                        <div class="row mt-3">
                            <div class="col-md-8"></div>
                            <div class="col-md-4">
                                <div class="card">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between">
                                            <span>Total:</span>
                                            <span id="editQuotationTotal">Ksh 0.00</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Update Quotation</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Custom JavaScript -->
    <script>
        // Enhanced Alert system with modern styling
        function showAlert(message, type = 'info') {
            const alertContainer = document.getElementById('alertContainer');
            const alertId = 'alert-' + Date.now();
            
            let alertClass = 'alert-info';
            let iconClass = 'fas fa-info-circle';
            
            if (type === 'success') {
                alertClass = 'alert-success';
                iconClass = 'fas fa-check-circle';
            } else if (type === 'error') {
                alertClass = 'alert-danger';
                iconClass = 'fas fa-exclamation-circle';
            } else if (type === 'warning') {
                alertClass = 'alert-warning';
                iconClass = 'fas fa-exclamation-triangle';
            }
            
            const alertHTML = `
                <div id="${alertId}" class="alert ${alertClass} alert-dismissible fade show" role="alert" style="border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.15);">
                    <div class="d-flex align-items-center">
                        <i class="${iconClass} me-3" style="font-size: 1.2rem;"></i>
                        <div class="flex-grow-1">
                            <strong>${type.charAt(0).toUpperCase() + type.slice(1)}!</strong> ${message}
                        </div>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                </div>
            `;
            
            alertContainer.insertAdjacentHTML('beforeend', alertHTML);
            
            // Auto-dismiss after 5 seconds
            setTimeout(() => {
                const alert = document.getElementById(alertId);
                if (alert) {
                    const bsAlert = new bootstrap.Alert(alert);
                    bsAlert.close();
                }
            }, 5000);
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, checking for action buttons...');
            
            // Test if action buttons exist
            const viewButtons = document.querySelectorAll('.view-quotation');
            const editButtons = document.querySelectorAll('.edit-quotation');
            const deleteButtons = document.querySelectorAll('.delete-quotation');
            
            console.log('Found buttons:', {
                view: viewButtons.length,
                edit: editButtons.length,
                delete: deleteButtons.length
            });
            
            // AJAX Search functionality
            const searchInput = document.getElementById('quotationSearchInput');
            if (searchInput) {
                let searchTimeout;
                searchInput.addEventListener('input', function() {
                    clearTimeout(searchTimeout);
                    const query = this.value.trim();
                    
                    // Show loading indicator
                    const tbody = document.querySelector('.table tbody');
                    if (query.length > 0) {
                        tbody.innerHTML = '<tr><td colspan="7" class="text-center"><i class="fas fa-spinner fa-spin"></i> Searching...</td></tr>';
                    }
                    
                    // Debounce search requests
                    searchTimeout = setTimeout(() => {
                        fetch(`../api/search_quotations.php?q=${encodeURIComponent(query)}`)
                            .then(res => res.json())
                            .then(data => {
                                if (data.success) {
                                    updateQuotationTable(data.quotations);
                                } else {
                                    tbody.innerHTML = '<tr><td colspan="7" class="text-center text-danger">Error: ' + (data.error || 'Search failed') + '</td></tr>';
                                }
                            })
                            .catch(error => {
                                tbody.innerHTML = '<tr><td colspan="7" class="text-center text-danger">Error: ' + error.message + '</td></tr>';
                            });
                    }, 300);
                });
            }
            
            function updateQuotationTable(quotations) {
                const tbody = document.querySelector('.table tbody');
                
                if (quotations.length === 0) {
                    tbody.innerHTML = '<tr><td colspan="7" class="text-center"><i class="fas fa-search fa-2x text-muted mb-2"></i><p class="text-muted">No quotations found matching your search.</p></td></tr>';
                    return;
                }
                
                tbody.innerHTML = quotations.map(quotation => `
                    <tr>
                        <td>
                            <strong>${escapeHtml(quotation.quote_number)}</strong>
                        </td>
                        <td>
                            <div>
                                <strong>${escapeHtml(quotation.customer_name || 'Walk-in Customer')}</strong>
                                <br>
                                <small class="text-muted">Created by ${escapeHtml(quotation.created_by_name || 'Unknown')}</small>
                            </div>
                        </td>
                        <td>
                            <strong class="text-primary">Ksh ${parseFloat(quotation.total_amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong>
                        </td>
                        <td>
                            <span class="badge ${getQuotationStatusBadgeClass(quotation.status)}">
                                <i class="fas fa-${getStatusIcon(quotation.status)} me-1"></i>
                                ${quotation.status.charAt(0).toUpperCase() + quotation.status.slice(1)}
                            </span>
                        </td>
                        <td>
                            <div>
                                <strong>${formatDate(quotation.created_at)}</strong>
                                <br>
                                <small class="text-muted">${formatTime(quotation.created_at)}</small>
                            </div>
                        </td>
                        <td>
                            <div>
                                <strong>${formatDate(quotation.valid_until)}</strong>
                                <br>
                                <small class="text-muted">${getDaysLeft(quotation.valid_until)}</small>
                            </div>
                        </td>
                        <td>
                            <div class="btn-group">
                                <button class="btn btn-outline-primary view-quotation" 
                                        title="View Quotation" 
                                        data-id="${quotation.id}" 
                                        data-quote-number="${escapeHtml(quotation.quote_number)}"
                                        data-bs-toggle="modal" 
                                        data-bs-target="#viewQuotationModal">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="btn btn-outline-info edit-quotation" 
                                        title="Edit Quotation"
                                        data-id="${quotation.id}" 
                                        data-quote-number="${escapeHtml(quotation.quote_number)}">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-outline-success convert-to-sale" 
                                        title="Convert to Sale"
                                        data-id="${quotation.id}" 
                                        data-quote-number="${escapeHtml(quotation.quote_number)}"
                                        data-total="${quotation.total_amount}"
                                        ${quotation.status !== 'active' ? 'disabled' : ''}>
                                    <i class="fas fa-arrow-right"></i>
                                </button>
                                <button class="btn btn-outline-warning schedule-delivery" 
                                        title="Schedule Delivery"
                                        data-id="${quotation.id}" 
                                        data-quote-number="${escapeHtml(quotation.quote_number)}"
                                        data-customer-id="${quotation.customer_id || ''}"
                                        data-customer-name="${escapeHtml(quotation.customer_name || 'Walk-in Customer')}"
                                        data-customer-phone="${escapeHtml(quotation.customer_phone || '')}"
                                        data-customer-address="${escapeHtml(quotation.customer_address || '')}"
                                        ${quotation.status !== 'active' ? 'disabled' : ''}>
                                    <i class="fas fa-truck"></i>
                                </button>
                                <button class="btn btn-outline-danger delete-quotation" 
                                        title="Delete Quotation"
                                        data-id="${quotation.id}" 
                                        data-quote-number="${escapeHtml(quotation.quote_number)}">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `).join('');
                
                // Re-attach event listeners for action buttons
                attachQuotationActionListeners();
            }
            
            function getStatusIcon(status) {
                switch(status) {
                    case 'active': return 'check-circle';
                    case 'expired': return 'times-circle';
                    case 'delivery_scheduled': return 'truck';
                    default: return 'clock';
                }
            }
            
            function formatTime(dateString) {
                return new Date(dateString).toLocaleTimeString('en-US', {
                    hour: 'numeric',
                    minute: '2-digit',
                    hour12: true
                });
            }
            
            function getDaysLeft(validUntil) {
                const daysLeft = (new Date(validUntil) - new Date()) / (1000 * 60 * 60 * 24);
                if (daysLeft > 0) {
                    return Math.round(daysLeft) + ' days left';
                } else {
                    return 'Expired';
                }
            }
            
            function getQuotationStatusBadgeClass(status) {
                switch(status) {
                    case 'active': return 'bg-success';
                    case 'expired': return 'bg-danger';
                    case 'converted': return 'bg-info';
                    case 'delivery_scheduled': return 'bg-warning';
                    default: return 'bg-secondary';
                }
            }
            
            function formatDate(dateString) {
                return new Date(dateString).toLocaleDateString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    year: 'numeric'
                });
            }
            
            function escapeHtml(text) {
                const div = document.createElement('div');
                div.textContent = text;
                return div.innerHTML;
            }
            
            function attachQuotationActionListeners() {
                // Re-attach view, edit, and delete event listeners for dynamically added rows
                // This will be handled by the existing document.addEventListener('click') handlers
            }
            
            // Set default valid until date (30 days from now)
            const validUntilInput = document.getElementById('valid_until');
            const thirtyDaysFromNow = new Date();
            thirtyDaysFromNow.setDate(thirtyDaysFromNow.getDate() + 30);
            validUntilInput.value = thirtyDaysFromNow.toISOString().split('T')[0];
            
            // Add item functionality
            const addItemBtn = document.getElementById('addItem');
            if (addItemBtn) {
                addItemBtn.addEventListener('click', function() {
                const quotationItems = document.getElementById('quotationItems');
                const newItem = quotationItems.querySelector('.quotation-item').cloneNode(true);
                
                // Clear values
                newItem.querySelectorAll('input, select').forEach(input => {
                    input.value = '';
                });
                
                quotationItems.appendChild(newItem);
                updateItemEventListeners();
            });
            }
            
            // Remove item functionality
            window.updateItemEventListeners = function() {
                document.querySelectorAll('.remove-item').forEach(btn => {
                    btn.addEventListener('click', function() {
                        if (document.querySelectorAll('.quotation-item').length > 1) {
                            this.closest('.quotation-item').remove();
                            calculateTotal();
                            
                            // Update scroll hint visibility
                            const itemCount = document.querySelectorAll('#quotationItems .quotation-item').length;
                            const scrollHint = document.getElementById('addScrollHint');
                            if (itemCount > 7) {
                                scrollHint.style.display = 'block';
                            } else {
                                scrollHint.style.display = 'none';
                            }
                        }
                    });
                });
                
                // Product selection change
                document.querySelectorAll('select[name="product_id[]"]').forEach(select => {
                    select.addEventListener('change', function() {
                        const row = this.closest('.quotation-item');
                        const priceInput = row.querySelector('input[name="unit_price[]"]');
                        const selectedOption = this.options[this.selectedIndex];
                        
                        if (selectedOption.dataset.price) {
                            priceInput.value = selectedOption.dataset.price;
                            calculateRowTotal(row);
                        }
                    });
                });
                
                // Quantity and price change
                document.querySelectorAll('input[name="quantity[]"], input[name="unit_price[]"]').forEach(input => {
                    input.addEventListener('input', function() {
                        calculateRowTotal(this.closest('.quotation-item'));
                    });
                });
            }
            
            function calculateRowTotal(row) {
                const quantity = parseFloat(row.querySelector('input[name="quantity[]"]').value) || 0;
                const unitPrice = parseFloat(row.querySelector('input[name="unit_price[]"]').value) || 0;
                const totalInput = row.querySelector('input[name="total_price[]"]');
                
                totalInput.value = (quantity * unitPrice).toFixed(2);
                calculateTotal();
            }
            
            window.calculateTotal = function() {
                let total = 0;
                document.querySelectorAll('input[name="total_price[]"]').forEach(input => {
                    total += parseFloat(input.value) || 0;
                });
                
                document.getElementById('quotationTotal').textContent = 'Ksh ' + total.toFixed(2);
            }
            
            // Initialize event listeners
            updateItemEventListeners();
            
            // Form submission
            const addQuotationForm = document.getElementById('addQuotationForm');
            if (addQuotationForm) {
                addQuotationForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                // Validate form
                const form = e.target;
                const formData = new FormData(form);
                
                // Validate that at least one product is selected
                const productSelects = form.querySelectorAll('select[name="product_id[]"]');
                let hasValidItems = false;
                
                productSelects.forEach(select => {
                    if (select.value) {
                        hasValidItems = true;
                    }
                });
                
                if (!hasValidItems) {
                    showAlert('Please select at least one product', 'error');
                    return;
                }
                
                // Show loading state
                const submitBtn = form.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creating...';
                
                // Submit to API
                fetch('../api/create_quotation.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showAlert(`Quotation ${data.quote_number} created successfully!`, 'success');
                        
                        // Reset form
                        form.reset();
                        
                        // Reset items to just one row
                        const quotationItems = document.getElementById('quotationItems');
                        quotationItems.innerHTML = `
                            <div class="row quotation-item">
                                <div class="col-md-5">
                                    <select class="form-select" name="product_id[]" required>
                                        <option value="">Select Product</option>
                                        <?php foreach ($products as $product): ?>
                                            <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price']; ?>">
                                                <?php echo htmlspecialchars($product['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="quantity[]" placeholder="Qty" min="1" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="unit_price[]" placeholder="Price" step="0.01" required>
                                </div>
                                <div class="col-md-2">
                                    <input type="number" class="form-control" name="total_price[]" placeholder="Total" readonly>
                                </div>
                                <div class="col-md-1">
                                    <button type="button" class="btn btn-outline-danger btn-sm remove-item">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        `;
                        
                        // Reset totals
                        document.getElementById('quotationTotal').textContent = 'Ksh 0.00';
                        
                        // Reset valid until date
                        const validUntilInput = document.getElementById('valid_until');
                        const thirtyDaysFromNow = new Date();
                        thirtyDaysFromNow.setDate(thirtyDaysFromNow.getDate() + 30);
                        validUntilInput.value = thirtyDaysFromNow.toISOString().split('T')[0];
                        
                        // Re-initialize event listeners
                        updateItemEventListeners();
                        
                        // Close modal
                bootstrap.Modal.getInstance(document.getElementById('addQuotationModal')).hide();
                        
                        // Refresh the page to show the new quotation
                        setTimeout(() => {
                            location.reload();
                        }, 1500);
                        
                    } else {
                        showAlert(data.message || 'Error creating quotation', 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showAlert('Network error. Please try again.', 'error');
                })
                .finally(() => {
                    // Reset button state
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                });
            });
            }
            
            // Action button handlers
            
            // View quotation handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.view-quotation')) {
                    console.log('View quotation clicked');
                    const btn = e.target.closest('.view-quotation');
                    const quotationId = btn.dataset.id;
                    const quoteNumber = btn.dataset.quoteNumber;
                    
                    console.log('Quotation ID:', quotationId, 'Quote Number:', quoteNumber);
                    
                    // Update modal title
                    document.querySelector('#viewQuotationModal .modal-title').textContent = 'View Quotation - ' + quoteNumber;
                    
                    // Load quotation details
                    fetch(`../api/get_quotation_details.php?id=${quotationId}`)
                        .then(response => response.json())
                        .then(data => {
                            console.log('API Response:', data);
                            if (data.success) {
                                displayQuotationDetails(data.quotation, data.items);
                            } else {
                                document.getElementById('quotationDetails').innerHTML = `
                                    <div class="alert alert-danger">
                                        <i class="fas fa-exclamation-circle"></i> ${data.message}
                                    </div>
                                `;
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            document.getElementById('quotationDetails').innerHTML = `
                                <div class="alert alert-danger">
                                    <i class="fas fa-exclamation-circle"></i> Error loading quotation details
                                </div>
                            `;
                        });
                }
            });
            
            // Delete quotation handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.delete-quotation')) {
                    console.log('Delete quotation clicked');
                    const btn = e.target.closest('.delete-quotation');
                    const quotationId = btn.dataset.id;
                    const quoteNumber = btn.dataset.quoteNumber;
                    
                    console.log('Delete - Quotation ID:', quotationId, 'Quote Number:', quoteNumber);
                    
                    if (confirm(`Are you sure you want to delete quotation ${quoteNumber}? This action cannot be undone.`)) {
                        // Show loading state
                        btn.disabled = true;
                        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
                        
                        // Send delete request
                        const formData = new FormData();
                        formData.append('id', quotationId);
                        
                        fetch('../api/delete_quotation.php', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => response.json())
                        .then(data => {
                            console.log('Delete API Response:', data);
                            if (data.success) {
                                showAlert(data.message, 'success');
                                // Remove the row from table
                                btn.closest('tr').remove();
                            } else {
                                showAlert(data.message, 'error');
                                // Reset button
                                btn.disabled = false;
                                btn.innerHTML = '<i class="fas fa-trash"></i>';
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            showAlert('Error deleting quotation', 'error');
                            // Reset button
                            btn.disabled = false;
                            btn.innerHTML = '<i class="fas fa-trash"></i>';
                        });
                    }
                }
            });
            
            // Convert to sale handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.convert-to-sale')) {
                    const btn = e.target.closest('.convert-to-sale');
                    const quotationId = btn.dataset.id;
                    const quoteNumber = btn.dataset.quoteNumber;
                    
                    if (confirm(`Convert quotation ${quoteNumber} to a sale? This will create a new sale record.`)) {
                        showAlert('Convert to sale functionality will be implemented soon', 'info');
                    }
                }
            });
            
            // Schedule delivery handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.schedule-delivery')) {
                    const btn = e.target.closest('.schedule-delivery');
                    const quotationId = btn.dataset.id;
                    const quoteNumber = btn.dataset.quoteNumber;
                    const customerId = btn.dataset.customerId;
                    const customerName = btn.dataset.customerName;
                    const customerPhone = btn.dataset.customerPhone;
                    const customerAddress = btn.dataset.customerAddress;
                    
                    if (confirm(`Schedule delivery for quotation ${quoteNumber}? This will redirect you to the delivery page.`)) {
                        // Store quotation data in sessionStorage for delivery page
                        sessionStorage.setItem('selectedQuotation', JSON.stringify({
                            id: quotationId,
                            quoteNumber: quoteNumber,
                            customerId: customerId,
                            customerName: customerName,
                            customerPhone: customerPhone,
                            customerAddress: customerAddress
                        }));
                        
                        // Redirect to delivery page
                        window.location.href = 'delivery.php';
                    }
                }
            });
            
            // Edit quotation handler
            document.addEventListener('click', function(e) {
                if (e.target.closest('.edit-quotation')) {
                    console.log('Edit quotation clicked');
                    const btn = e.target.closest('.edit-quotation');
                    const quotationId = btn.dataset.id;
                    const quoteNumber = btn.dataset.quoteNumber;
                    
                    console.log('Edit - Quotation ID:', quotationId, 'Quote Number:', quoteNumber);
                    
                    // Update modal title
                    document.querySelector('#editQuotationModal .modal-title').textContent = 'Edit Quotation - ' + quoteNumber;
                    
                    // Load quotation details for editing
                    loadQuotationForEdit(quotationId);
                    
                    // Show edit modal
                    const editModal = new bootstrap.Modal(document.getElementById('editQuotationModal'));
                    editModal.show();
                }
            });
            
            // Function to display quotation details in modal
            function displayQuotationDetails(quotation, items) {
                const customerInfo = quotation.customer_name ? `
                    <div class="row">
                        <div class="col-md-6">
                            <strong>Customer:</strong> ${quotation.customer_name}<br>
                            <strong>Phone:</strong> ${quotation.customer_phone || 'N/A'}<br>
                            <strong>Email:</strong> ${quotation.customer_email || 'N/A'}
                        </div>
                        <div class="col-md-6">
                            <strong>Address:</strong><br>
                            ${quotation.customer_address || 'N/A'}
                        </div>
                    </div>
                ` : '<p><strong>Customer:</strong> Walk-in Customer</p>';
                
                let itemsHtml = '';
                items.forEach(item => {
                    itemsHtml += `
                        <tr>
                            <td>${item.product_name}</td>
                            <td>${item.quantity}</td>
                            <td>Ksh ${parseFloat(item.unit_price).toFixed(2)}</td>
                            <td>Ksh ${parseFloat(item.total_price).toFixed(2)}</td>
                        </tr>
                    `;
                });
                
                const html = `
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <h6>Quotation Information</h6>
                            <p><strong>Quote Number:</strong> ${quotation.quote_number}</p>
                            <p><strong>Status:</strong> <span class="badge ${quotation.status === 'active' ? 'bg-success' : (quotation.status === 'expired' ? 'bg-danger' : (quotation.status === 'delivery_scheduled' ? 'bg-info' : 'bg-warning'))}">${quotation.status.replace('_', ' ').charAt(0).toUpperCase() + quotation.status.replace('_', ' ').slice(1)}</span></p>
                            <p><strong>Created:</strong> ${new Date(quotation.created_at).toLocaleDateString()}</p>
                            <p><strong>Valid Until:</strong> ${new Date(quotation.valid_until).toLocaleDateString()}</p>
                            <p><strong>Created By:</strong> ${quotation.created_by_name}</p>
                        </div>
                        <div class="col-md-6">
                            <h6>Customer Information</h6>
                            ${customerInfo}
                        </div>
                    </div>
                    
                    ${quotation.notes ? `
                        <div class="mb-3">
                            <h6>Notes</h6>
                            <p class="text-muted">${quotation.notes}</p>
                        </div>
                    ` : ''}
                    
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h6 class="mb-0">Items (${items.length} items)</h6>
                        ${items.length > 9 ? '<small class="text-muted">Scroll to see all items</small>' : ''}
                    </div>
                    
                    <!-- Fixed Header -->
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Product</th>
                                    <th>Quantity</th>
                                    <th>Unit Price</th>
                                    <th>Total</th>
                                </tr>
                            </thead>
                        </table>
                    </div>
                    
                    <!-- Scrollable Body -->
                    <div class="table-responsive" style="max-height: ${items.length > 9 ? '300px' : 'auto'}; overflow-y: ${items.length > 9 ? 'auto' : 'visible'};">
                        <table class="table table-sm">
                            <tbody>
                                ${itemsHtml}
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="mt-3 p-2 bg-light rounded" style="border-left: 4px solid var(--primary-color);">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="mb-0 text-primary">Total Amount</h6>
                            <h5 class="mb-0 font-weight-bold text-primary">Ksh ${parseFloat(quotation.total_amount).toFixed(2)}</h5>
                        </div>
                    </div>
                `;
                
                document.getElementById('quotationDetails').innerHTML = html;
            }
            
            // Print quotation handler
            const printQuotationBtn = document.getElementById('printQuotation');
            if (printQuotationBtn) {
                printQuotationBtn.addEventListener('click', function() {
                const quotationDetails = document.getElementById('quotationDetails');
                if (quotationDetails.innerHTML.includes('Loading')) {
                    showAlert('Please wait for quotation details to load', 'warning');
                    return;
                }
                
                // Get store settings for print header (including logo and website)
                const storeSettings = {
                    name: '<?php echo htmlspecialchars($settings['store_name'] ?? 'POS System'); ?>',
                    address: '<?php echo htmlspecialchars($settings['store_address'] ?? ''); ?>',
                    phone: '<?php echo htmlspecialchars($settings['store_phone'] ?? ''); ?>',
                    email: '<?php echo htmlspecialchars($settings['store_email'] ?? ''); ?>',
                    website: '<?php echo htmlspecialchars($settings['website'] ?? ''); ?>',
                    logo: '<?php 
                        // Get logo from database
                        $stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
                        $stmt->execute();
                        $logo = $stmt->fetch();
                        if ($logo && $logo['setting_binary']) {
                            $mimeType = 'image/jpeg'; // Default to JPEG
                            if (strpos($logo['setting_value'], '.png') !== false) {
                                $mimeType = 'image/png';
                            } elseif (strpos($logo['setting_value'], '.gif') !== false) {
                                $mimeType = 'image/gif';
                            }
                            echo 'data:' . $mimeType . ';base64,' . base64_encode($logo['setting_binary']);
                        }
                    ?>'
                };
                
                // Create print window
                const printWindow = window.open('', '_blank', 'width=800,height=600');
                
                // Get current quotation data from modal
                const modalTitle = document.querySelector('#viewQuotationModal .modal-title').textContent;
                
                // Extract quotation data from the modal content for proper print layout
                const quotationInfo = extractQuotationDataFromModal();
                
                // Debug: Log the extracted data to console
                console.log('Extracted quotation data:', quotationInfo);
                
                // Print HTML template with proper layout for all items
                const printHtml = `
                    <!DOCTYPE html>
                    <html>
                    <head>
                        <title>${modalTitle}</title>
                        <style>
                            body { 
                                font-family: Arial, sans-serif; 
                                margin: 20px; 
                                color: #333; 
                                line-height: 1.4;
                            }
                            .header { 
                                text-align: center; 
                                border-bottom: 2px solid #333; 
                                padding-bottom: 20px; 
                                margin-bottom: 30px; 
                            }
                            .header .logo { 
                                max-width: 150px; 
                                max-height: 80px; 
                                margin-bottom: 15px; 
                                border-radius: 8px;
                                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                            }
                            .header h1 { 
                                margin: 10px 0; 
                                font-size: 24px; 
                                color: #333; 
                            }
                            .header p { 
                                margin: 5px 0; 
                                font-size: 14px; 
                                color: #666; 
                            }
                            .header .website { 
                                color: #0066cc; 
                                text-decoration: none; 
                                font-weight: 500;
                            }
                            .quotation-title { 
                                font-size: 20px; 
                                font-weight: bold; 
                                text-align: center; 
                                margin: 20px 0; 
                                color: #333; 
                            }
                            .info-section {
                                margin-bottom: 25px;
                            }
                            .info-row { 
                                display: flex; 
                                margin-bottom: 15px; 
                                flex-wrap: wrap;
                            }
                            .info-col { 
                                flex: 1; 
                                min-width: 300px;
                                padding-right: 20px; 
                            }
                            .info-col h6 {
                                font-size: 16px; 
                                margin: 0 0 10px 0; 
                                color: #333;
                                border-bottom: 1px solid #ddd;
                                padding-bottom: 5px;
                            }
                            .info-col p {
                                margin: 5px 0; 
                                font-size: 14px;
                            }
                            .table { 
                                width: 100%; 
                                border-collapse: collapse; 
                                margin: 20px 0; 
                                page-break-inside: auto;
                            }
                            .table th, .table td { 
                                border: 1px solid #ddd; 
                                padding: 10px 8px; 
                                text-align: left; 
                                vertical-align: top;
                            }
                            .table th { 
                                background-color: #f8f9fa; 
                                font-weight: bold; 
                                font-size: 14px;
                            }
                            .table td {
                                font-size: 13px;
                            }
                            .table tbody tr {
                                page-break-inside: avoid;
                                page-break-after: auto;
                            }
                            .badge { 
                                padding: 4px 8px; 
                                border-radius: 4px; 
                                font-size: 12px; 
                                font-weight: bold; 
                            }
                            .bg-success { 
                                background-color: #28a745; 
                                color: white; 
                            }
                            .bg-danger { 
                                background-color: #dc3545; 
                                color: white; 
                            }
                            .bg-warning { 
                                background-color: #ffc107; 
                                color: black; 
                            }
                            .bg-info { 
                                background-color: #17a2b8; 
                                color: white; 
                            }
                            .text-muted { 
                                color: #6c757d; 
                            }
                            .text-primary {
                                color: #007bff;
                            }
                            .total-section {
                                margin-top: 30px;
                                padding: 15px;
                                background-color: #f8f9fa;
                                border-left: 4px solid #007bff;
                                border-radius: 5px;
                            }
                            .total-row {
                                display: flex;
                                justify-content: space-between;
                                align-items: center;
                                font-size: 16px;
                            }
                            .total-label {
                                font-weight: bold;
                                color: #007bff;
                            }
                            .total-amount {
                                font-weight: bold;
                                font-size: 18px;
                                color: #007bff;
                            }
                            .footer { 
                                margin-top: 50px; 
                                text-align: center; 
                                font-size: 12px; 
                                color: #666; 
                                page-break-inside: avoid;
                            }
                            .notes-section {
                                margin: 20px 0;
                                padding: 15px;
                                background-color: #f8f9fa;
                                border-radius: 5px;
                                border-left: 3px solid #ffc107;
                            }
                            .notes-section h6 {
                                margin: 0 0 10px 0;
                                color: #856404;
                            }
                            @media print {
                                body { 
                                    margin: 15px; 
                                    font-size: 12px;
                                }
                                .header { 
                                    margin-bottom: 20px; 
                                    page-break-inside: avoid;
                                }
                                .header .logo { 
                                    max-width: 120px; 
                                    max-height: 60px; 
                                }
                                .info-section {
                                    page-break-inside: avoid;
                                }
                                .table {
                                    font-size: 11px;
                                }
                                .table th, .table td {
                                    padding: 6px 4px;
                                }
                                .total-section {
                                    page-break-inside: avoid;
                                }
                                .footer {
                                    page-break-inside: avoid;
                                }
                            }
                        </style>
                    </head>
                    <body>
                        <div class="header">
                            ${storeSettings.logo ? `<img src="${storeSettings.logo}" alt="Store Logo" class="logo">` : ''}
                            <h1>${storeSettings.name}</h1>
                            ${storeSettings.address ? `<p>${storeSettings.address}</p>` : ''}
                            ${storeSettings.phone ? `<p>Phone: ${storeSettings.phone}</p>` : ''}
                            ${storeSettings.email ? `<p>Email: ${storeSettings.email}</p>` : ''}
                            ${storeSettings.website ? `<p>Website: <span class="website">${storeSettings.website}</span></p>` : ''}
                        </div>
                        
                        <div class="quotation-title">QUOTATION</div>
                        
                        <div class="content">
                            <div class="info-section">
                                <div class="info-row">
                                    <div class="info-col">
                                        <h6>Quotation Information</h6>
                                        <p><strong>Quote Number:</strong> ${quotationInfo.quoteNumber}</p>
                                        <p><strong>Status:</strong> <span class="badge ${quotationInfo.statusClass}">${quotationInfo.statusText}</span></p>
                                        <p><strong>Created:</strong> ${quotationInfo.createdDate}</p>
                                        <p><strong>Valid Until:</strong> ${quotationInfo.validUntil}</p>
                                        <p><strong>Created By:</strong> ${quotationInfo.createdBy}</p>
                                    </div>
                                    <div class="info-col">
                                        <h6>Customer Information</h6>
                                        ${quotationInfo.customerInfo}
                                    </div>
                                </div>
                            </div>
                            
                            ${quotationInfo.notes ? `
                                <div class="notes-section">
                                    <h6>Notes</h6>
                                    <p class="text-muted">${quotationInfo.notes}</p>
                                </div>
                            ` : ''}
                            
                            <div class="table-section">
                                <h6 style="margin-bottom: 15px;">Items (${quotationInfo.items.length} items)</h6>
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th style="width: 50%;">Product</th>
                                            <th style="width: 15%;">Quantity</th>
                                            <th style="width: 20%;">Unit Price</th>
                                            <th style="width: 15%;">Total</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        ${quotationInfo.items.map(item => `
                                            <tr>
                                                <td>${item.product_name}</td>
                                                <td>${item.quantity}</td>
                                                <td>Ksh ${parseFloat(item.unit_price).toFixed(2)}</td>
                                                <td>Ksh ${parseFloat(item.total_price).toFixed(2)}</td>
                                            </tr>
                                        `).join('')}
                                    </tbody>
                                </table>
                            </div>
                            
                            <div class="total-section">
                                <div class="total-row">
                                    <span class="total-label">Total Amount</span>
                                    <span class="total-amount">Ksh ${parseFloat(quotationInfo.totalAmount).toFixed(2)}</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="footer">
                            <p>Thank you for your business!</p>
                            <p>Generated on: ${new Date().toLocaleDateString()} at ${new Date().toLocaleTimeString()}</p>
                        </div>
                    </body>
                    </html>
                `;
                
                // Write to print window and print
                printWindow.document.write(printHtml);
                printWindow.document.close();
                
                // Wait for content to load then print
                printWindow.onload = function() {
                    printWindow.print();
                    setTimeout(() => {
                        printWindow.close();
                    }, 1000);
                };
            });
        }
        
        // Function to extract quotation data from modal for proper print layout
        function extractQuotationDataFromModal() {
            const modalContent = document.getElementById('quotationDetails');
            
            // Helper function to find element by text content
            function findElementByText(selector, text) {
                const elements = modalContent.querySelectorAll(selector);
                for (let element of elements) {
                    if (element.textContent.includes(text)) {
                        return element;
                    }
                }
                return null;
            }
            
            // Extract basic information
            const quoteNumberElement = findElementByText('p', 'Quote Number');
            const quoteNumber = quoteNumberElement ? quoteNumberElement.textContent.split(':')[1]?.trim() : 'N/A';
            
            const statusElement = modalContent.querySelector('.badge');
            const statusText = statusElement ? statusElement.textContent.trim() : 'Unknown';
            const statusClass = statusElement ? statusElement.className.match(/bg-\w+/)?.[0] || 'bg-secondary' : 'bg-secondary';
            
            // Extract dates
            const createdElement = findElementByText('p', 'Created');
            const createdDate = createdElement ? createdElement.textContent.split(':')[1]?.trim() : 'N/A';
            
            const validUntilElement = findElementByText('p', 'Valid Until');
            const validUntil = validUntilElement ? validUntilElement.textContent.split(':')[1]?.trim() : 'N/A';
            
            const createdByElement = findElementByText('p', 'Created By');
            const createdBy = createdByElement ? createdByElement.textContent.split(':')[1]?.trim() : 'N/A';
            
            // Extract customer information
            const customerCols = modalContent.querySelectorAll('.col-md-6');
            const customerInfo = customerCols.length > 1 ? customerCols[1].innerHTML : '<p><strong>Customer:</strong> Walk-in Customer</p>';
            
            // Extract notes
            const notesElement = modalContent.querySelector('p.text-muted');
            const notes = notesElement ? notesElement.textContent.trim() : '';
            
            // Extract items from the table
            const items = [];
            const itemRows = modalContent.querySelectorAll('tbody tr');
            itemRows.forEach(row => {
                const cells = row.querySelectorAll('td');
                if (cells.length >= 4) {
                    items.push({
                        product_name: cells[0].textContent.trim(),
                        quantity: cells[1].textContent.trim(),
                        unit_price: cells[2].textContent.replace('Ksh ', '').trim(),
                        total_price: cells[3].textContent.replace('Ksh ', '').trim()
                    });
                }
            });
            
            // Extract total amount - try multiple selectors to find the total
            let totalAmount = '0.00';
            
            // Method 1: Look for h5 with text-primary class
            let totalElement = modalContent.querySelector('h5.text-primary');
            if (totalElement) {
                totalAmount = totalElement.textContent.replace('Ksh ', '').trim();
            } else {
                // Method 2: Look for any element with text-primary class
                totalElement = modalContent.querySelector('.text-primary');
                if (totalElement) {
                    totalAmount = totalElement.textContent.replace('Ksh ', '').trim();
                } else {
                    // Method 3: Look for text containing "Total Amount" and extract the amount
                    const totalSection = findElementByText('div', 'Total Amount');
                    if (totalSection) {
                        const amountText = totalSection.textContent.match(/Ksh\s*([\d,]+\.?\d*)/);
                        if (amountText) {
                            totalAmount = amountText[1].replace(',', '');
                        }
                    } else {
                        // Method 4: Calculate total from items as fallback
                        let calculatedTotal = 0;
                        items.forEach(item => {
                            calculatedTotal += parseFloat(item.total_price) || 0;
                        });
                        totalAmount = calculatedTotal.toFixed(2);
                    }
                }
            }
            
            return {
                quoteNumber,
                statusText,
                statusClass,
                createdDate,
                validUntil,
                createdBy,
                customerInfo,
                notes,
                items,
                totalAmount
            };
        }
         
         // Load quotation data for editing
         function loadQuotationForEdit(quotationId) {
             fetch(`../api/get_quotation_details.php?id=${quotationId}`)
                 .then(response => response.json())
                 .then(data => {
                     if (data.success) {
                         populateEditForm(data.quotation, data.items);
                     } else {
                         showAlert(data.message, 'error');
                     }
                 })
                 .catch(error => {
                     console.error('Error:', error);
                     showAlert('Error loading quotation details', 'error');
                 });
         }
         
         // Populate edit form with quotation data
         function populateEditForm(quotation, items) {
             document.getElementById('edit_quotation_id').value = quotation.id;
             document.getElementById('edit_customer_id').value = quotation.customer_id || '';
             document.getElementById('edit_valid_until').value = quotation.valid_until;
             document.getElementById('edit_notes').value = quotation.notes || '';
             
             // Clear existing items
             const itemsContainer = document.getElementById('editQuotationItems');
             itemsContainer.innerHTML = '';
             
             // Add items
             items.forEach(item => {
                 addEditItem(item);
             });
             
             // Show/hide scroll hint based on item count
             const scrollHint = document.getElementById('editScrollHint');
             if (items.length > 7) {
                 scrollHint.style.display = 'block';
             } else {
                 scrollHint.style.display = 'none';
             }
             
             // Calculate total
             calculateEditTotal();
         }
         
         // Add item to edit form (global scope)
         window.addEditItem = function(itemData = null) {
             const itemsContainer = document.getElementById('editQuotationItems');
             
             const itemDiv = document.createElement('div');
             itemDiv.className = 'row quotation-item mb-2';
             
             const productOptions = `
                 <option value="">Select Product</option>
                 <?php foreach ($products as $product): ?>
                     <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price']; ?>">
                         <?php echo htmlspecialchars($product['name']); ?>
                     </option>
                 <?php endforeach; ?>
             `;
             
             itemDiv.innerHTML = `
                 <div class="col-md-5">
                     <select class="form-select" name="product_id[]" required>
                         ${productOptions}
                     </select>
                 </div>
                 <div class="col-md-2">
                     <input type="number" class="form-control" name="quantity[]" placeholder="Qty" min="1" 
                         value="${itemData ? itemData.quantity : ''}" required>
                 </div>
                 <div class="col-md-2">
                     <input type="number" class="form-control" name="unit_price[]" placeholder="Price" step="0.01" 
                         value="${itemData ? itemData.unit_price : ''}" required>
                 </div>
                 <div class="col-md-2">
                     <input type="number" class="form-control" name="total_price[]" placeholder="Total" readonly
                         value="${itemData ? itemData.total_price : ''}">
                 </div>
                 <div class="col-md-1">
                     <button type="button" class="btn btn-outline-danger btn-sm remove-edit-item">
                         <i class="fas fa-trash"></i>
                     </button>
                 </div>
             `;
             
             itemsContainer.appendChild(itemDiv);
             
             // Set selected product if itemData exists
             if (itemData) {
                 const productSelect = itemDiv.querySelector('select[name="product_id[]"]');
                 productSelect.value = itemData.product_id;
             }
             
             updateEditItemEventListeners();
             
             // Update scroll hint visibility
             const itemCount = document.querySelectorAll('#editQuotationItems .quotation-item').length;
             const scrollHint = document.getElementById('editScrollHint');
             if (itemCount > 7) {
                 scrollHint.style.display = 'block';
             } else {
                 scrollHint.style.display = 'none';
             }
         }
         

         
         // Update event listeners for edit items (global scope)
         window.updateEditItemEventListeners = function() {
             // Remove item buttons
             document.querySelectorAll('.remove-edit-item').forEach(btn => {
                 btn.addEventListener('click', function() {
                     if (document.querySelectorAll('#editQuotationItems .quotation-item').length > 1) {
                         this.closest('.quotation-item').remove();
                         calculateEditTotal();
                         
                         // Update scroll hint visibility
                         const itemCount = document.querySelectorAll('#editQuotationItems .quotation-item').length;
                         const scrollHint = document.getElementById('editScrollHint');
                         if (itemCount > 7) {
                             scrollHint.style.display = 'block';
                         } else {
                             scrollHint.style.display = 'none';
                         }
                     }
                 });
             });
             
             // Product selection change
             document.querySelectorAll('#editQuotationItems select[name="product_id[]"]').forEach(select => {
                 select.addEventListener('change', function() {
                     const row = this.closest('.quotation-item');
                     const priceInput = row.querySelector('input[name="unit_price[]"]');
                     const selectedOption = this.options[this.selectedIndex];
                     
                     if (selectedOption.dataset.price) {
                         priceInput.value = selectedOption.dataset.price;
                         calculateEditRowTotal(row);
                     }
                 });
             });
             
             // Quantity and price change
             document.querySelectorAll('#editQuotationItems input[name="quantity[]"], #editQuotationItems input[name="unit_price[]"]').forEach(input => {
                 input.addEventListener('input', function() {
                     calculateEditRowTotal(this.closest('.quotation-item'));
                 });
             });
         }
         
         // Calculate row total for edit form (global scope)
         window.calculateEditRowTotal = function(row) {
             const quantity = parseFloat(row.querySelector('input[name="quantity[]"]').value) || 0;
             const unitPrice = parseFloat(row.querySelector('input[name="unit_price[]"]').value) || 0;
             const totalInput = row.querySelector('input[name="total_price[]"]');
             
             totalInput.value = (quantity * unitPrice).toFixed(2);
             calculateEditTotal();
         }
         
         // Calculate total for edit form (global scope)
         window.calculateEditTotal = function() {
             let total = 0;
             document.querySelectorAll('#editQuotationItems input[name="total_price[]"]').forEach(input => {
                 total += parseFloat(input.value) || 0;
             });
             
             document.getElementById('editQuotationTotal').textContent = 'Ksh ' + total.toFixed(2);
         }
         
         // Live search functionality for edit modal
         let searchTimeout;
         const editProductSearch = document.getElementById('editProductSearch');
         const editSearchResults = document.getElementById('editSearchResults');
         
         if (editProductSearch) {
             editProductSearch.addEventListener('input', function() {
                 clearTimeout(searchTimeout);
                 const query = this.value.trim();
                 
                 if (query.length < 2) {
                     editSearchResults.style.display = 'none';
                     return;
                 }
                 
                 // Show loading
                 editSearchResults.innerHTML = '<div class="p-2 text-center"><i class="fas fa-spinner fa-spin"></i> Searching...</div>';
                 editSearchResults.style.display = 'block';
                 
                 // Debounce search
                 searchTimeout = setTimeout(() => {
                     fetch(`../api/search_products.php?term=${encodeURIComponent(query)}`)
                         .then(response => response.json())
                         .then(data => {
                             if (data.success && data.products.length > 0) {
                                 const resultsHtml = data.products.map(product => `
                                     <div class="search-result-item p-2 border-bottom" 
                                          style="cursor: pointer; transition: background-color 0.2s;"
                                          data-product-id="${product.id}"
                                          data-product-name="${product.name.replace(/"/g, '&quot;')}"
                                          data-product-price="${product.selling_price}">
                                         <div class="d-flex justify-content-between align-items-center">
                                             <div>
                                                 <strong>${product.name}</strong>
                                                 <br>
                                                 <small class="text-muted">Product ID: ${product.id}</small>
                                             </div>
                                             <div class="text-end">
                                                 <strong class="text-primary">Ksh ${parseFloat(product.selling_price).toFixed(2)}</strong>
                                                 <br>
                                                 <small class="text-muted">Stock: ${product.stock_quantity || 0}</small>
                                             </div>
                                         </div>
                                     </div>
                                 `).join('');
                                 
                                 editSearchResults.innerHTML = resultsHtml;
                             } else {
                                 editSearchResults.innerHTML = '<div class="p-2 text-center text-muted">No products found</div>';
                             }
                         })
                         .catch(error => {
                             console.error('Search error:', error);
                             editSearchResults.innerHTML = '<div class="p-2 text-center text-danger">Error searching products</div>';
                         });
                 }, 300);
             });
             
             // Hide results when clicking outside
             if (editProductSearch && editSearchResults) {
                 document.addEventListener('click', function(e) {
                     // Don't hide if clicking on search results
                     if (editSearchResults.contains(e.target)) {
                         return;
                     }
                     
                     if (!editProductSearch.contains(e.target)) {
                         editSearchResults.style.display = 'none';
                     }
                 });
             }
             
             // Handle keyboard navigation
             if (editProductSearch) {
                 editProductSearch.addEventListener('keydown', function(e) {
                     if (e.key === 'Escape') {
                         if (editSearchResults) {
                             editSearchResults.style.display = 'none';
                         }
                         this.blur();
                     }
                 });
             }
             
             // Handle clicks on search results using event delegation
             if (editSearchResults) {
                 editSearchResults.addEventListener('click', function(e) {
                     const searchResultItem = e.target.closest('.search-result-item');
                     if (searchResultItem) {
                         const productId = searchResultItem.dataset.productId;
                         const productName = searchResultItem.dataset.productName;
                         const productPrice = parseFloat(searchResultItem.dataset.productPrice);
                         
                         addProductToEdit(productId, productName, productPrice);
                     }
                 });
                 
                 // Handle hover effects for search results
                 editSearchResults.addEventListener('mouseover', function(e) {
                     const searchResultItem = e.target.closest('.search-result-item');
                     if (searchResultItem) {
                         searchResultItem.style.backgroundColor = '#f8f9fa';
                     }
                 });
                 
                 editSearchResults.addEventListener('mouseout', function(e) {
                     const searchResultItem = e.target.closest('.search-result-item');
                     if (searchResultItem) {
                         searchResultItem.style.backgroundColor = 'white';
                     }
                 });
             }
         }
         
         // Function to add product from search to edit form (global scope)
         window.addProductToEdit = function(productId, productName, productPrice) {
             // Clear search
             if (editProductSearch) {
                 editProductSearch.value = '';
             }
             if (editSearchResults) {
                 editSearchResults.style.display = 'none';
             }
             
             // Add new item with the selected product
             const itemData = {
                 product_id: productId,
                 quantity: 1,
                 unit_price: productPrice,
                 total_price: productPrice
             };
             
             addEditItem(itemData);
             
             // Show success message
             showAlert(`Added ${productName} to quotation`, 'success');
         };
         
         // Live search functionality for add modal
         let addSearchTimeout;
         const addProductSearch = document.getElementById('addProductSearch');
         const addSearchResults = document.getElementById('addSearchResults');
         
         if (addProductSearch) {
             addProductSearch.addEventListener('input', function() {
                 clearTimeout(addSearchTimeout);
                 const query = this.value.trim();
                 
                 if (query.length < 2) {
                     addSearchResults.style.display = 'none';
                     return;
                 }
                 
                 // Show loading
                 addSearchResults.innerHTML = '<div class="p-2 text-center"><i class="fas fa-spinner fa-spin"></i> Searching...</div>';
                 addSearchResults.style.display = 'block';
                 
                 // Debounce search
                 addSearchTimeout = setTimeout(() => {
                     fetch(`../api/search_products.php?term=${encodeURIComponent(query)}`)
                         .then(response => response.json())
                         .then(data => {
                             if (data.success && data.products.length > 0) {
                                 const resultsHtml = data.products.map(product => `
                                     <div class="search-result-item p-2 border-bottom" 
                                          style="cursor: pointer; transition: background-color 0.2s;"
                                          data-product-id="${product.id}"
                                          data-product-name="${product.name.replace(/"/g, '&quot;')}"
                                          data-product-price="${product.selling_price}">
                                         <div class="d-flex justify-content-between align-items-center">
                                             <div>
                                                 <strong>${product.name}</strong>
                                                 <br>
                                                 <small class="text-muted">Product ID: ${product.id}</small>
                                             </div>
                                             <div class="text-end">
                                                 <strong class="text-primary">Ksh ${parseFloat(product.selling_price).toFixed(2)}</strong>
                                                 <br>
                                                 <small class="text-muted">Stock: ${product.stock_quantity || 0}</small>
                                             </div>
                                         </div>
                                     </div>
                                 `).join('');
                                 
                                 addSearchResults.innerHTML = resultsHtml;
                             } else {
                                 addSearchResults.innerHTML = '<div class="p-2 text-center text-muted">No products found</div>';
                             }
                         })
                         .catch(error => {
                             console.error('Search error:', error);
                             addSearchResults.innerHTML = '<div class="p-2 text-center text-danger">Error searching products</div>';
                         });
                 }, 300);
             });
             
             // Hide results when clicking outside
             if (addProductSearch && addSearchResults) {
                 document.addEventListener('click', function(e) {
                     // Don't hide if clicking on search results
                     if (addSearchResults.contains(e.target)) {
                         return;
                     }
                     
                     if (!addProductSearch.contains(e.target)) {
                         addSearchResults.style.display = 'none';
                     }
                 });
             }
             
             // Handle keyboard navigation
             if (addProductSearch) {
                 addProductSearch.addEventListener('keydown', function(e) {
                     if (e.key === 'Escape') {
                         if (addSearchResults) {
                             addSearchResults.style.display = 'none';
                         }
                         this.blur();
                     }
                 });
             }
             
             // Handle clicks on search results using event delegation
             if (addSearchResults) {
                 addSearchResults.addEventListener('click', function(e) {
                     const searchResultItem = e.target.closest('.search-result-item');
                     if (searchResultItem) {
                         const productId = searchResultItem.dataset.productId;
                         const productName = searchResultItem.dataset.productName;
                         const productPrice = parseFloat(searchResultItem.dataset.productPrice);
                         
                         addProductToAdd(productId, productName, productPrice);
                     }
                 });
                 
                 // Handle hover effects for search results
                 addSearchResults.addEventListener('mouseover', function(e) {
                     const searchResultItem = e.target.closest('.search-result-item');
                     if (searchResultItem) {
                         searchResultItem.style.backgroundColor = '#f8f9fa';
                     }
                 });
                 
                 addSearchResults.addEventListener('mouseout', function(e) {
                     const searchResultItem = e.target.closest('.search-result-item');
                     if (searchResultItem) {
                         searchResultItem.style.backgroundColor = 'white';
                     }
                 });
             }
         }
         
         // Function to add product from search to add form (global scope)
         window.addProductToAdd = function(productId, productName, productPrice) {
             // Clear search
             if (addProductSearch) {
                 addProductSearch.value = '';
             }
             if (addSearchResults) {
                 addSearchResults.style.display = 'none';
             }
             
             // Add new item with the selected product
             const itemData = {
                 product_id: productId,
                 quantity: 1,
                 unit_price: productPrice,
                 total_price: productPrice
             };
             
             addItemWithData(itemData);
             
             // Show success message
             showAlert(`Added ${productName} to quotation`, 'success');
         };
         
         // Function to add item with data to add form (global scope)
         window.addItemWithData = function(itemData = null) {
             const quotationItems = document.getElementById('quotationItems');
             const newItem = quotationItems.querySelector('.quotation-item').cloneNode(true);
             
             // Clear values
             newItem.querySelectorAll('input, select').forEach(input => {
                 input.value = '';
             });
             
             // Set values if itemData exists
             if (itemData) {
                 const productSelect = newItem.querySelector('select[name="product_id[]"]');
                 const quantityInput = newItem.querySelector('input[name="quantity[]"]');
                 const priceInput = newItem.querySelector('input[name="unit_price[]"]');
                 const totalInput = newItem.querySelector('input[name="total_price[]"]');
                 
                 productSelect.value = itemData.product_id;
                 quantityInput.value = itemData.quantity;
                 priceInput.value = itemData.unit_price;
                 totalInput.value = itemData.total_price;
             }
             
             quotationItems.appendChild(newItem);
             updateItemEventListeners();
             calculateTotal();
             
             // Update scroll hint visibility
             const itemCount = document.querySelectorAll('#quotationItems .quotation-item').length;
             const scrollHint = document.getElementById('addScrollHint');
             if (itemCount > 7) {
                 scrollHint.style.display = 'block';
             } else {
                 scrollHint.style.display = 'none';
             }
         }
         
         // Edit form submission
         const editQuotationForm = document.getElementById('editQuotationForm');
         if (editQuotationForm) {
             editQuotationForm.addEventListener('submit', function(e) {
             e.preventDefault();
             
             // Validate form
             const form = e.target;
             const formData = new FormData(form);
             
             // Validate that at least one product is selected
             const productSelects = form.querySelectorAll('select[name="product_id[]"]');
             let hasValidItems = false;
             
             productSelects.forEach(select => {
                 if (select.value) {
                     hasValidItems = true;
                 }
             });
             
             if (!hasValidItems) {
                 showAlert('Please select at least one product', 'error');
                 return;
             }
             
             // Show loading state
             const submitBtn = form.querySelector('button[type="submit"]');
             const originalText = submitBtn.textContent;
             submitBtn.disabled = true;
             submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
             
             // Submit to API
             fetch('../api/update_quotation.php', {
                 method: 'POST',
                 body: formData
             })
             .then(response => response.json())
             .then(data => {
                 if (data.success) {
                     showAlert(data.message, 'success');
                     
                     // Close modal
                     bootstrap.Modal.getInstance(document.getElementById('editQuotationModal')).hide();
                     
                     // Refresh the page to show updated quotation
                     setTimeout(() => {
                         location.reload();
                     }, 1500);
                     
                 } else {
                     showAlert(data.message || 'Error updating quotation', 'error');
                 }
             })
             .catch(error => {
                 console.error('Error:', error);
                 showAlert('Network error. Please try again.', 'error');
             })
             .finally(() => {
                 // Reset button state
                 submitBtn.disabled = false;
                 submitBtn.textContent = originalText;
             });
             });
         }
        });
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 